// Popup script - The Coordinator
document.addEventListener('DOMContentLoaded', () => {
    document.getElementById('scrapeBtn').addEventListener('click', startScrapeProcess);
    document.getElementById('copyBtn').addEventListener('click', copyToClipboard);

    // Auto-scrape on open
    startScrapeProcess();
});

async function startScrapeProcess() {
    showStatus('Initializing...', 'info');

    try {
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });

        if (!tab.url.includes('fantasy.espn.com')) {
            showWrongPageError();
            return;
        }

        // 1. Get Authentication Cookies (Robustly)
        showStatus('Fetching cookies...', 'info');
        const cookieData = await getEspnCookies(tab.url);

        if (!cookieData.swid || !cookieData.espn_s2) {
            showStatus('⚠️ Missing ESPN cookies. Please log in.', 'warning');
            // Proceed anyway, might fail but user sees error
        }

        // 2. Inject & Scrape Page Data
        showStatus('Scraping page data...', 'info');
        const pageData = await scrapePageData(tab.id);

        if (!pageData) {
            // Already handled by scrapePageData error UI
            return;
        }

        // 3. Merge Data
        const finalData = {
            leagueId: pageData.leagueId,
            seasonId: pageData.seasonId || '2026',
            swid: cookieData.swid,
            espn_s2: cookieData.espn_s2,
            scoring: pageData.scoring || {}
        };

        // 4. Display & Validate
        displayData(finalData, pageData);

    } catch (err) {
        console.error(err);
        showStatus('❌ Error: ' + err.message, 'error');
    }
}

function getEspnCookies(url) {
    return new Promise((resolve) => {
        chrome.cookies.getAll({ domain: '.espn.com' }, (cookies) => {
            const data = { swid: null, espn_s2: null };
            cookies.forEach(c => {
                if (c.name === 'SWID') data.swid = c.value;
                if (c.name === 'espn_s2') data.espn_s2 = decodeURIComponent(c.value);
            });
            resolve(data);
        });
    });
}

function scrapePageData(tabId) {
    return new Promise((resolve, reject) => {
        // Retry logic wrapper
        const trySendMessage = (retries = 3) => {
            chrome.tabs.sendMessage(tabId, { action: 'scrape' }, (response) => {
                if (chrome.runtime.lastError) {
                    if (retries > 0) {
                        console.log('Injecting script...');
                        chrome.scripting.executeScript({
                            target: { tabId: tabId },
                            files: ['content.js']
                        }, () => {
                            if (chrome.runtime.lastError) {
                                showStatus('❌ Injection failed. Refresh page.', 'error');
                                resolve(null);
                            } else {
                                // Wait for script to initialize
                                setTimeout(() => trySendMessage(retries - 1), 200);
                            }
                        });
                    } else {
                        showStatus('❌ Connection failed. Refresh page.', 'error');
                        resolve(null);
                    }
                } else {
                    // Check if response is valid
                    if (!response) {
                        showStatus('❌ Scraper returned no data.', 'error');
                        resolve(null);
                    } else {
                        // Support both wrapped { data: ... } and direct object
                        resolve(response.data || response);
                    }
                }
            });
        };

        trySendMessage();
    });
}

function displayData(finalData, pageData) {
    const jsonStr = JSON.stringify(finalData, null, 2);
    document.getElementById('dataOutput').value = jsonStr;

    // Validation Logic
    const hasScoring = finalData.scoring && Object.keys(finalData.scoring).length > 0;
    const isSettingsPage = pageData.isSettingsPage; // Now returned by content.js

    if (hasScoring && isSettingsPage) {
        showStatus('✅ Success! Copied cookies & scoring.', 'success');
    } else if (hasScoring && !isSettingsPage) {
        // We scraped something, but we aren't sure it's correct because we aren't on the settings page.
        // Safer to ask user to verify.
        showStatus('⚠️ Scraped data, but verify scoring!', 'warning');
        showScoringGuide(finalData.leagueId, finalData.seasonId);
    } else {
        // Guide the user
        showStatus('⚠️ Scoring settings missing.', 'warning');
        showScoringGuide(finalData.leagueId, finalData.seasonId);
    }
}

function showWrongPageError() {
    showStatus('⚠️ Not on ESPN Fantasy.', 'warning');
    const container = document.getElementById('statusMsg');
    container.innerHTML += `
        <div style="margin-top: 10px;">
            <a href="https://fantasy.espn.com/basketball" target="_blank" class="btn-guide">Go to ESPN Fantasy 🏀</a>
        </div>
    `;
}

function showScoringGuide(leagueId, seasonId) {
    if (!leagueId) return;
    const container = document.getElementById('statusMsg');
    container.innerHTML += `
        <div style="margin-top: 10px; padding: 10px; background: #fff3cd; border-radius: 4px; border: 1px solid #ffeeba; color: #856404; font-size: 12px;">
            <strong>To get scoring data:</strong><br/>
            1. Open Scoring Settings below.<br/>
            2. Open this popup again.<br/>
            <a href="https://fantasy.espn.com/basketball/league/settings?leagueId=${leagueId}&seasonId=${seasonId}&view=scoring" 
               target="_blank" class="btn-guide" style="margin-top: 6px;">
               Open Scoring Settings ⚙️
            </a>
        </div>
    `;
}

function copyToClipboard() {
    const text = document.getElementById('dataOutput').value;
    if (!text) return;
    navigator.clipboard.writeText(text).then(() => {
        showStatus('✅ Copied to clipboard!', 'success');
        setTimeout(() => showStatus('✅ Success! Copied cookies & scoring.', 'success'), 2000);
    });
}

function showStatus(msg, type) {
    const el = document.getElementById('statusMsg');
    el.className = 'status ' + type;
    el.textContent = msg;
}
