// Content script - The ESPN Fantasy Scraper
console.log('ESPN Fantasy Scraper loaded');

// Helper to reliably get seasonId from URL or Path
function getSeasonId() {
    // 1. Try URL parameters
    const urlParams = new URLSearchParams(window.location.search);
    if (urlParams.has('seasonId')) {
        return urlParams.get('seasonId');
    }

    // 2. Try Path Match (e.g. /fantasy/basketball/season/2026/...)
    // Adjust regex for various ESPN patters
    const pathMatch = window.location.pathname.match(/\/season\/(\d{4})/);
    if (pathMatch) {
        return pathMatch[1];
    }

    const leagueMatch = window.location.pathname.match(/fba\.(\d{4})\.l\./); // e.g. fba.2023.l.1234
    if (leagueMatch) {
        return leagueMatch[1];
    }

    // 3. Fallback: Current Year logic
    const now = new Date();
    // If month is > July (August+), we are likely in the 'next' year's season context (e.g. Oct 2024 is Season 2025)
    return now.getMonth() > 6 ? (now.getFullYear() + 1).toString() : now.getFullYear().toString();
}

// Helper to reliably get leagueId
function getLeagueId() {
    // 1. Try URL parameters
    const urlParams = new URLSearchParams(window.location.search);
    if (urlParams.has('leagueId')) {
        return urlParams.get('leagueId');
    }

    // 2. Try Hash Params
    if (window.location.hash) {
        const hashParams = new URLSearchParams(window.location.hash.substring(1));
        if (hashParams.has('leagueId')) {
            return hashParams.get('leagueId');
        }
    }

    // 3. Try Path Match
    const pathMatch = window.location.pathname.match(/\/league\/(\d+)/);
    if (pathMatch) {
        return pathMatch[1];
    }

    const leaguePathMatch = window.location.pathname.match(/\.l\.(\d+)/); // e.g. fba.2023.l.123456
    if (leaguePathMatch) {
        return leaguePathMatch[1];
    }

    return null;
}

// Scrape scoring settings from Text/Table views (Regular User View)
function findScoringInText() {
    const scoring = {};
    const statPatterns = {
        'Points': 'PTS', 'PTS': 'PTS', 'Pts': 'PTS',
        'Rebounds': 'REB', 'REB': 'REB', 'Reb': 'REB',
        'Assists': 'AST', 'AST': 'AST', 'Ast': 'AST',
        'Steals': 'STL', 'STL': 'STL', 'Stl': 'STL',
        'Blocks': 'BLK', 'BLK': 'BLK', 'Blk': 'BLK',
        'Turnovers': 'TO', 'TO': 'TO', 'Turnover': 'TO',
        'Field Goals Made': 'FGM', 'FGM': 'FGM',
        'Field Goals Attempted': 'FGA', 'FGA': 'FGA',
        'Free Throws Made': 'FTM', 'FTM': 'FTM',
        'Free Throws Attempted': 'FTA', 'FTA': 'FTA',
        '3-Pointers Made': '3PM', '3PM': '3PM', 'Three Pointers Made': '3PM',
        '3-Pointers Attempted': '3PA', '3PA': '3PA',
        'Offensive Rebounds': 'OREB', 'Offensive Rebound': 'OREB',
        'Defensive Rebounds': 'DREB', 'Defensive Rebound': 'DREB',
        'Double Doubles': 'DD', 'Double-Double': 'DD',
        'Triple Doubles': 'TD', 'Triple-Double': 'TD',
        'Quadruple Doubles': 'QD', 'Quadruple-Double': 'QD'
    };

    // Strategy 0: Explicit ESPN "Scoring Setting Row" (Common in View Mode)
    // Structure: <div class="scoring-setting-row"><div class="setting-cell">Name</div><div class="setting-cell">Value</div></div>
    const specificRows = document.querySelectorAll('.scoring-setting-row');
    if (specificRows.length > 0) {
        specificRows.forEach(row => {
            const cells = row.querySelectorAll('.setting-cell');
            if (cells.length >= 2) {
                const nameText = cells[0].textContent.trim();
                const valText = cells[1].textContent.trim(); // The underlying value

                for (const [name, key] of Object.entries(statPatterns)) {
                    if (nameText.includes(name)) {
                        const val = parseFloat(valText);
                        if (!isNaN(val)) {
                            scoring[key] = val;
                        }
                    }
                }
            }
        });
    }

    // Strategy 1: General Table Rows (Fallback/Standard View)
    const rows = document.querySelectorAll('tr, .stat-row, .Table__TR');
    rows.forEach(row => {
        // Skip if we already processed this via specificRows (optimization, implementation optional but good)
        // Check if this row contains a specificRow we already handled?
        if (row.querySelector('.scoring-setting-row')) return;

        const text = row.innerText || "";
        const cleanText = text.replace(/\s+/g, ' ').trim();

        for (const [name, key] of Object.entries(statPatterns)) {
            if (cleanText.includes(name)) {
                // Determine value. Usually in the last cell or adjacent cell.
                const cells = row.querySelectorAll('td, .Table__TD');
                let foundVal = null;

                if (cells.length > 0) {
                    // Start from end, look for numeric value
                    for (let i = cells.length - 1; i >= 0; i--) {
                        const cellText = cells[i].textContent.trim();
                        // Check if pure number (ignoring empty strings)
                        if (cellText && !isNaN(parseFloat(cellText))) {
                            foundVal = parseFloat(cellText);
                            break;
                        }
                    }
                } else {
                    // Try regex on row text if no cells found (e.g. div rows)
                    const matches = cleanText.match(/([\d.-]+)$/); // Number at end?
                    if (matches) {
                        foundVal = parseFloat(matches[1]);
                    }
                }

                if (foundVal !== null && !isNaN(foundVal)) {
                    scoring[key] = foundVal;
                }
            }
        }
    });

    return scoring;
}

// Scrape scoring from Input fields (Commissioner View context)
function findScoringInInputs() {
    const inputScoring = {};
    const statPatterns = {
        'Points': 'PTS', 'PTS': 'PTS',
        'Rebounds': 'REB', 'REB': 'REB',
        'Assists': 'AST', 'AST': 'AST',
        'Steals': 'STL', 'STL': 'STL',
        'Blocks': 'BLK', 'BLK': 'BLK',
        'Turnovers': 'TO', 'TO': 'TO',
        'Field Goals Made': 'FGM', 'FGM': 'FGM',
        'Field Goals Attempted': 'FGA', 'FGA': 'FGA',
        'Free Throws Made': 'FTM', 'FTM': 'FTM',
        'Free Throws Attempted': 'FTA', 'FTA': 'FTA',
        '3-Pointers Made': '3PM', '3PM': '3PM', 'Three Pointers Made': '3PM',
        '3-Pointers Attempted': '3PA', '3PA': '3PA'
    };

    const rows = document.querySelectorAll('tr, .stat-row');
    rows.forEach(row => {
        const text = row.innerText;
        for (const [name, key] of Object.entries(statPatterns)) {
            if (text.includes(name)) {
                const input = row.querySelector('input');
                if (input) {
                    const val = parseFloat(input.value);
                    if (!isNaN(val)) {
                        inputScoring[key] = val;
                    }
                }
            }
        }
    });
    return inputScoring;
}

// Fallback: Scan entire body text for "Stat Value" patterns
// Useful if table structure allows identification (e.g. CSS grid) or is obfuscated
function scanPageTextFallback() {
    const scoring = {};
    const bodyText = document.body.innerText.replace(/\s+/g, ' '); // Flatten newlines

    // We iterate patterns and look for "Name [Value]" or "Name: [Value]"
    // This is aggressive and might have false positives, but it's a fallback.

    /* Re-declare patterns map for local use or move to shared scope */
    const statPatterns = {
        'Points': 'PTS', 'PTS': 'PTS', 'Pts': 'PTS',
        'Rebounds': 'REB', 'REB': 'REB', 'Reb': 'REB',
        'Assists': 'AST', 'AST': 'AST', 'Ast': 'AST',
        'Steals': 'STL', 'STL': 'STL', 'Stl': 'STL',
        'Blocks': 'BLK', 'BLK': 'BLK', 'Blk': 'BLK',
        'Turnovers': 'TO', 'TO': 'TO', 'Turnover': 'TO',
        'Field Goals Made': 'FGM', 'FGM': 'FGM',
        'Field Goals Attempted': 'FGA', 'FGA': 'FGA',
        'Free Throws Made': 'FTM', 'FTM': 'FTM',
        'Free Throws Attempted': 'FTA', 'FTA': 'FTA',
        '3-Pointers Made': '3PM', '3PM': '3PM', 'Three Pointers Made': '3PM',
        '3-Pointers Attempted': '3PA', '3PA': '3PA',
        'Offensive Rebounds': 'OREB', 'Offensive Rebound': 'OREB',
        'Defensive Rebounds': 'DREB', 'Defensive Rebound': 'DREB',
        'Double Doubles': 'DD', 'Double-Double': 'DD',
        'Triple Doubles': 'TD', 'Triple-Double': 'TD',
        'Quadruple Doubles': 'QD', 'Quadruple-Double': 'QD'
    };

    for (const [name, key] of Object.entries(statPatterns)) {
        // Regex: 
        // 1. The Name (escaped)
        // 2. Optional colon or space
        // 3. The Number (captured)
        // We use 'i' for case insensitivity
        const escapedName = name.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
        const regex = new RegExp(`${escapedName}\\s*[:\\-]?\\s*([+\\-]?\\d+(\\.\\d+)?)`, 'i');

        const match = bodyText.match(regex);
        if (match) {
            const val = parseFloat(match[1]);
            if (!isNaN(val)) {
                // Simple heuristic: Fantasy params usually aren't huge years like 2024 (unless default)
                // But let's trust the match for now
                scoring[key] = val;
            }
        }
    }
    return scoring;
}

// Main Extraction Function
function extractLeagueData() {
    const leagueId = getLeagueId();
    const seasonId = getSeasonId();

    // URL Parsing for page type
    const isSettingsPage = window.location.href.includes('settings') ||
        window.location.href.includes('view=scoring') ||
        window.location.href.includes('scoring');

    const data = {
        leagueId: leagueId,
        seasonId: seasonId,
        scoringPeriodId: null, // Hard to scrape reliably without deeper logic
        scoring: {},
        isSettingsPage: isSettingsPage
    };

    if (!leagueId) {
        console.warn('Could not extract leagueId from URL');
        // We return data anyway, popup handles missing ID error
    }

    // Try Commissioner View (Inputs) first
    const inputs = document.querySelectorAll('input[type="text"][value], input[type="number"][value]');
    if (inputs.length > 5) {
        data.scoring = findScoringInInputs();
    }

    // If no scoring found (or not in inputs view), try Text Table View
    if (Object.keys(data.scoring).length === 0) {
        data.scoring = findScoringInText();
    }

    // FINAL FALLBACK: Text Regex Scan
    if (Object.keys(data.scoring).length === 0) {
        console.log('Structured scraping failed. Using text fallback.');
        data.scoring = scanPageTextFallback();
    }

    return data;
}

// Message Listener
if (!window.hasEspnScraperListener) {
    window.hasEspnScraperListener = true;
    chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
        if (request.action === 'scrape') {
            const data = extractLeagueData();
            console.log('Scraped Data:', data);
            sendResponse(data);
        }
        return true;
    });
}
